{-# LANGUAGE RankNTypes #-}

data Expr a = Lit Int
            | Add (Expr a) (Expr a)
            | IfZero (Expr a) (Expr a) (Expr a)
            | Var a
            | Let (Expr a) (a -> Expr a)
            | Lam (a -> Expr a)
            | App (Expr a) (Expr a)
            | Mu (a -> Expr a)

type ClosedExpr = forall a. Expr a

n k = Lit k
plus = Add
apply = App
let_ e1 e2 = Let e1 (\x -> e2 (Var x))
lam_ e = Lam (\x -> e (Var x))
mu_ e = Mu (\x -> e (Var x))
letrec_ e1 e2 = Let (Mu (\x -> e1 (Var x))) (\x -> e2 (Var x))

data Value = N Int | F (Value -> Value)
instance Show Value where
    show (N n) = show n
    show (F _) = "fun"

eval :: Expr Value -> Value
eval (Lit n) = N n
eval (Add e1 e2) = add (eval e1) (eval e2)
eval (IfZero e1 e2 e3) = ifZero (eval e1) (eval e2) (eval e3)
eval (Var x) = x
eval (Let e1 e2) = eval (e2 (eval e1))
eval (Lam e) = F (\v -> eval (e v))
eval (App e1 e2) = app (eval e1) (eval e2)
eval (Mu e) = fix (\v -> eval (e v))

add (N m) (N n) = N (m + n)
ifZero (N n) v1 v2 = if n == 0 then v1 else v2
app (F f) v = f v
fix :: (a -> a) -> a
fix f = let r = f r in r

-- example 1
example1 =  let_ (lam_ (\x -> x `plus` Lit (-1))) (\dec ->
            let_ (lam_ (\f -> lam_ (\x -> f `apply` (f `apply` x)))) (\twice ->
            (twice `apply` twice `apply` dec `apply` (Lit 10))))
-- this encodes:
--let dec x = x - 1
--    twice f x = f (f x)
--in  twice twice dec 10

-- example 2 - multiplication
mul :: ClosedExpr
mul = lam_ (\m -> mu_ (\rec -> lam_ (\n ->
        IfZero n (Lit 0) (m `plus` (rec `apply` (n `plus` Lit (-1)))))))
-- this encodes:
--let mul m =
--  let rec n =
--      if n == 0
--          then 0
--          else m + (rec (n - 1))
--      in rec

-- example 3 - factorial
fact :: ClosedExpr
fact = letrec_ (\rec -> lam_ (\n ->
        IfZero n (Lit 1)
        (mul `apply` n `apply` (rec `apply` (n `plus` Lit (-1))))
      ))
      (\rec -> lam_ (\m -> rec `apply` m))
        

main = do
      putStrLn $ show $ eval (n 10)
      putStrLn $ show $ eval $ (n 10) `plus` (n 20)
      putStrLn $ show $ eval $ example1
      putStrLn $ show $ eval $ mul `apply` (n 6) `apply` (n 3)
      putStrLn $ show $ eval $ fact `apply` (n 5)

