/* error.c: communication of informational, warning, error and fatal error
 * messages. */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include "brep.h"

/* The default message callback routines ignore the client data */
/* default callback for showing informational messages */
static void brep_info(void *client_data, char *message)
{
#ifndef DEBUG
  /* don't show purely informational messages */
#else /*DEBUG*/
  fprintf(stderr, "%s\n", message);
#endif /*DEBUG*/
}

/* default callback for showing warning messages */
static void brep_warning(void *client_data, char *message)
{
  fprintf(stderr, "warning: %s\n", message);
}

/* default callback for showing error messages */
static void brep_error(void *client_data, char *message)
{
  fprintf(stderr, "error: %s\n", message);
}

/* default callback for showing fatal error messages */
static void brep_fatal(void *client_data, char *message)
{
  fprintf(stderr, "fatal error: %s\n", message);
  exit(-1);
}

static BREP_MSG_CALLBACK_FUNC brep_info_callback_func = brep_info,
                              brep_warning_callback_func = brep_warning,
                              brep_error_callback_func = brep_error,
                              brep_fatal_callback_func = brep_fatal;

/* The SetCallback functions return the previously set callback function */
BREP_MSG_CALLBACK_FUNC BrepSetInfoCallback(BREP_MSG_CALLBACK_FUNC func)
{
  BREP_MSG_CALLBACK_FUNC old_func = brep_info_callback_func;
  brep_info_callback_func = func;
  return old_func;
}

BREP_MSG_CALLBACK_FUNC BrepSetWarningCallback(BREP_MSG_CALLBACK_FUNC func)
{
  BREP_MSG_CALLBACK_FUNC old_func = brep_warning_callback_func;
  brep_warning_callback_func = func;
  return old_func;
}

BREP_MSG_CALLBACK_FUNC BrepSetErrorCallback(BREP_MSG_CALLBACK_FUNC func)
{
  BREP_MSG_CALLBACK_FUNC old_func = brep_error_callback_func;
  brep_error_callback_func = func;
  return old_func;
}

BREP_MSG_CALLBACK_FUNC BrepSetFatalCallback(BREP_MSG_CALLBACK_FUNC func)
{
  BREP_MSG_CALLBACK_FUNC old_func = brep_fatal_callback_func;
  brep_fatal_callback_func = func;
  return old_func;
}

/* prints an informational message */
void BrepInfo(void *client_data, char *routine, char *text, ...)
{
	va_list pvar;
	char buf[BREP_MAX_MESSAGE_LENGTH];
	
	va_start(pvar, text);
	vsprintf(buf, text, pvar);
	va_end(pvar);

#ifdef DEBUG
	if (routine) {
	  char *p;
	  /* go to the end of the string */
	  for (p=buf; *p; p++) {}
	  sprintf(p, " (%s)", routine);
	}
#endif
	
	brep_info_callback_func(client_data, buf);
}

/* prints a warning message */
void BrepWarning(void *client_data, char *routine, char *text, ...)
{
	va_list pvar;
	char buf[BREP_MAX_MESSAGE_LENGTH];

	va_start(pvar, text);
	vsprintf(buf, text, pvar);
	va_end(pvar);

#ifdef DEBUG
	if (routine) {
	  char *p;
	  /* go to the end of the string */
	  for (p=buf; *p; p++) {}
	  sprintf(p, " (%s)", routine);
	}
#endif
	
	brep_warning_callback_func(client_data, buf);
}

/* prints an error message */
void BrepError(void *client_data, char *routine, char *text, ...)
{
	va_list pvar;
	char buf[BREP_MAX_MESSAGE_LENGTH];
	
	va_start(pvar, text);
	vsprintf(buf, text, pvar);
	va_end(pvar);

#ifdef DEBUG
	if (routine) {
	  char *p;
	  /* go to the end of the string */
	  for (p=buf; *p; p++) {}
	  sprintf(p, " (%s)", routine);
	}
#endif
	
	brep_error_callback_func(client_data, buf);
}

/* prints a fatal error message */
void BrepFatal(void *client_data, char *routine, char *text, ...)
{
	va_list pvar;
	char buf[BREP_MAX_MESSAGE_LENGTH];
	
	va_start(pvar, text);
	vsprintf(buf, text, pvar);
	va_end(pvar);

#ifdef DEBUG
	if (routine) {
	  char *p;
	  /* go to the end of the string */
	  for (p=buf; *p; p++) {}
	  sprintf(p, " (%s)", routine);
	}
#endif
	
	brep_fatal_callback_func(client_data, buf);
}

