/* brep_private.h: brep.c and friends private data structures and routines */

#ifndef _BREP_PRIVATE_H_
#define _BREP_PRIVATE_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "brep.h"

/* common first fields of BREP_SHELL, ... */
typedef struct BREP_RING {
  void *client_data;
  struct BREP_RING *prev, *next;
} BREP_RING;

/* iterators over a ring such as BREP_SHELL, ... */
extern void BrepIterate(BREP_RING *ring, void (*func)(BREP_RING *));
extern void BrepIterate1A(BREP_RING *ring, void (*func)(BREP_RING *, void *), void *parm);
extern void BrepIterate2A(BREP_RING *ring, void (*func)(BREP_RING *, void *, void *), void *parm1, void *parm2);

/* prints an informational, warning, error, fatal error  message */
extern void BrepInfo(void *client_data, char *routine, char *text, ...);
extern void BrepWarning(void *client_data, char *routine, char *text, ...);
extern void BrepError(void *client_data, char *routine, char *text, ...);
extern void BrepFatal(void *client_data, char *routine, char *text, ...);

/* various actions to be performed when a shell is specified completely */
extern void BrepCloseShell(BREP_SHELL *shell);

/* various actions to be performed when a face is specified completely */
extern void BrepCloseFace(BREP_FACE *face);

/* various actions to be performed when a contour is specified completely */
extern void BrepCloseContour(BREP_CONTOUR *contour);

/* various actions to be performed when an edge has been specified 
 * completely, i.e. it is enclosed in two contours. */
extern void BrepCloseEdge(BREP_EDGE *edge);

/* various actions to be performed when a contour has been updated */
extern void BrepUpdateContour(BREP_CONTOUR *contour);

/* Creates an edge connecting vertex1 to vertex2. It is not
 * connected to a contour. It calls the CreateEdge callback
 * if set. */
extern BREP_EDGE *BrepCreateEdge(BREP_VERTEX *vertex1, BREP_VERTEX *vertex2, void *client_data);

/* Creates a wing from vertex1 to vertex2. First looks for an
 * existing edge connecting the vertices. Creates such an edge if 
 * if there is not yet one. Returns a pointer to an unused wing in
 * the edge. If there is no unused wing in this edge, complains, and
 * creates a second edge between the two vertices. 
 * The wing still needs to be connected in a contour. */
extern BREP_WING *BrepCreateWingWithoutContour(BREP_VERTEX *vertex1, BREP_VERTEX *vertex2, void *client_data);

/* connects the shell to the solid */
extern void BrepConnectShellToSolid(BREP_SHELL *shell, BREP_SOLID *solid);

/* disconnect the shell from the solid */
extern void BrepDisconnectShellFromSolid(BREP_SHELL *shell);

/* connects the face to the shell */
extern void BrepConnectFaceToShell(BREP_FACE *face, BREP_SHELL *shell);

/* disconnect the face from its containing shell */
extern void BrepDisconnectFaceFromShell(BREP_FACE *face);

/* connects a contour to the specified face */
extern void BrepConnectContourToFace(BREP_CONTOUR *contour, BREP_FACE *face);

/* disconnects a contour from its containing face */
extern void BrepDisconnectContourFromFace(BREP_CONTOUR *contour);

/* connect the wing as last wing in the contour */
extern void BrepConnectWingToContour(BREP_WING *wing, BREP_CONTOUR *contour);

/* Disconnect the wing from the contour. Use with care: the contour
 * might not be a loop anymore after disconnecting the wing! */
extern void BrepDisconnectWingFromContour(BREP_WING *wing);

/* Connect the wing to its starting vertex. The wing must have been connected 
 * to its contour before. */
extern void BrepConnectWingToVertex(BREP_WING *wing);

/* Disconnect the wing from its starting vertex. The wing must be properly 
 * connected to a contour. */
extern void BrepDisconnectWingFromVertex(BREP_WING *wing);

/* release all storage associated with an edge if not used anymore in
 * any contour (the given edge must already be disconnected from its contours.) */
extern void BrepDestroyEdge(BREP_EDGE *edge);

#ifdef __cplusplus
}
#endif

#endif /*_BREP_PRIVATE_H_*/
