// Impact.h -- Particle impact implementation.


#ifndef IMPACT_H
#define IMPACT_H


#include <stdio.h>
#include <stdlib.h>
#include "vector2d.h"
#include "CompressedColor.h"



// Options for setOption method (can be ORed)
#define IMPACT_HAS_COORDINATES  1          // Save coordinates of impact
#define IMPACT_HAS_DIRECTION    2          // Save direction of impact
#define IMPACT_HAS_WEIGHTRGB    4          // Save weight of the particle
#define IMPACT_HAS_ID           8          // Save id (integer number)

 
// This class encodes particle impact in a memory efficient manner.
// You can choose which information you want to store using the
// setOptions method. This is a global setting.
class Impact
{
public:
  // Constructor.
  // OUT: NULL if memory allocation failed.
  Impact();

  // Specify storage option.
  // IN: A combinaison of options (see above)
  // POST: When a particle will be saved in an output stream or read in an input stream,
  //       the information saved/read is the one specified by flags.
  static void setOptions(const int flags);

  // Get the number of bytes needed to save one impact. 
  // OUT: Space needed on disk in bytes to save one impact.
  static int getNumBytes();

  // Specify/get patch id
  void setPatchId(const int pid);
  int getPatchId();

  // Specify/get coordinates of impact
  // COND: Parametric coordinates uv on a patch ::  0.0<=uv.u<=1.0, 0.0<=uv.v<=1.0
  // The information is packed in a constant precision format. Resolution = 2^-16
  void setCoordinates(const VEC2D& uv);
  void getCoordinates(VEC2D& uv);

  // Specify direction of impact
  // COND: 0.0<=phitheta.u<=2*PI, 0.0<=phitheta.v<=PI/2
  // The information is packed in a constant precision format. Resolution = 2*PI*2^-16 and PI*2-17
  void setDirection(const VEC2D& phitheta);
  void getDirection(VEC2D& phitheta);

  // Specify weight of the particle at the impact
  // COND: every component is >0.0
  void setWeight(const COLOR& col);
  void getWeight(COLOR& col);

  // Read the impact from InputStream
  // The information is read according to the last setOptions.
  void read(FILE*);

  // Write to OutputStream
  // The information is written according to the last setOptions.
  void write(FILE*);

private:
  int                    id;           // patch id
  unsigned short         u,v;          // impact coordinates
  unsigned short         p,t;          // impact direction
  CompressedColor        weightRGB;    // particle weight at the three standard wavelength
};

#endif



