/* mrvisibility.h: Multi-Resolution Visibility ala Sillion & Drettakis, "Multi-Resolution
 * Control of Visibility Error", SIGGRAPH '95 p145 */

#ifndef _MRVISIBILITY_H_
#define _MRVISIBILITY_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "geomlist.h"
#include "ray.h"

/* Following routines return a floating point number in the range [0..1]. 
 * 0 indicates that there is
 * full occlusion, 1 that there is full visibility and a number in between 
 * that there is visibility with at least one occluder with feature size 
 * smaller than the specified minimum feature size. (Such occluders have been 
 * replaced by a box containing an isotropic participating medium with
 * suitable extinction properties.) rcvdist is the distance from the origin of
 * the ray (assumed to be on the source) to the receiver surface. srcsize
 * is the diameter of the source surface. minfeaturesize is the minimal 
 * diameter of a feature (umbra or whole lit region on the receiver) 
 * that one is interested in. Approximate visibility computations are allowed 
 * for occluders that cast features with diameter smaller than 
 * minfeaturesize. If there is a "hard" occlusion, the first patch tested that
 * lead to this conclusion is added to the shadow cache (see shadowcaching.h). */

extern double GeomMultiResolutionVisibility(GEOM *geom, RAY *ray, float rcvdist, float srcsize, float minfeaturesize);

extern double GeomListMultiResolutionVisibility(GEOMLIST *occluderlist, RAY *ray, float rcvdist, float srcsize, float minfeaturesize);


/* Equivalent blocker size determination: first call BlockerInit(),
 * then call GeomBlcokerSize() of GeomBlockserSizeInDirection() for the
 * geoms for which you like to compute the equivalent blocker size, and
 * finally terminate with BlockerTerminate(). */

extern void BlockerInit(void);
extern void BlockerTerminate(void);

/* Determines the equivalent blocker size of the GEOM when viewed in the given
 * direction. */
extern double GeomBlcokerSizeInDirection(GEOM *geom, VECTOR viewdir);

/* Determines a single equivalent blocker size for the GEOM which can be used 
 * for any direction (approximately). Actually, it computes the maximum
 * blocker size in a number of directions. */
extern double GeomBlcokerSize(GEOM *geom);

#ifdef __cplusplus
}
#endif

#endif /*_MRVISIBILITY_H_*/
