/* render.h: interface to the graphics library for rendering.
 * graphics library dependent routines are in render.c, which is a symbolic
 * link to gl.c or another .c file. Graphics library independent routines
 * are in rendercommon.c. */

#ifndef _RENDER_H_
#define _RENDER_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <Xm/Xm.h>
#include "color.h"
#include "vectortype.h"
#include "bounds.h"

/* This function determines whether or not a non-default X visual and/or
 * colormap is needed to render in the canvas window on the given screen. If so,
 * the function returns True and fills in the XVIsualInfo and Colormap. If not,
 * the function doesn't fill in the XVisualInfo and Colormap and returns False. */
extern Boolean RenderGetVisualInfoAndColormap(Screen *screen, 
					      XVisualInfo *visual_info,
					      Colormap *colormap);

/* create a window for rendering into. Returns the created widget. */
extern Widget RenderCreateWindow(Widget parent);

#ifdef NEVER
/* renders the whole scene */
extern void RenderScene(void);

/* Renders an image of m lines of n pixels at column x on row y (= lower
 * left corner of image, relative to the lower left corner of the window) */
extern void RenderPixels(int x, int y, int n, int m, RGB *rgb);

/* sets the current color for line or outline drawing */
extern void RenderSetColor(RGB *rgb);

/* renders a convex polygon flat shaded in the current color */
extern void RenderPolygonFlat(int nrverts, POINT *verts);

/* renders a convex polygon with Gouraud shading */
extern void RenderPolygonGouraud(int nrverts, POINT *verts, RGB *vertcols);

/* renders a line from point p to point q, for eg debugging */
extern void RenderLine(VECTOR *p, VECTOR *q);

  /* renders a point (small circle, at the specified location */
extern void RenderPoint(VECTOR *p);

/* renders a bounding box. */
extern void RenderBounds(BOUNDINGBOX bounds);

/* indicates that the scene has modified, so a new display list should be
 * compiled and rendered from now on. Only relevant when using display lists. */
extern void RenderNewDisplayList(void);

/* returns only after all issued graphics commands have been processed.
 * RenderScene() already does so, but this function is needed in other
 * circumstances, such as when selecting a patch. */
extern void RenderFinish(void);
#endif /*NEVER*/

/* saves the image in the front buffer to a ppm file, viewable with e.g. xv */
extern void SaveScreen(FILE *fp);

extern int RenderStartFrame(void);
extern void RenderEndFrame(void);

typedef enum {BC_OFF, BC_ON, BC_DYNAMIC} BACKFACECULLINGMODE;

/* rendering options */
typedef struct RENDEROPTIONS {
  char	draw_outlines,	/* True for drawing facet outlines */
    smooth_shading,	/* True for rendering with Gouraud interpolation */
    draw_bounding_boxes,/* True for showing bounding boxes */
    draw_clusters,	/* True for showing cluster hierarchy */
    use_display_lists, 	/* true for using display lists for faster display */
    disable_lighting,   /* exactly what it says */
    ifs_hack,
    enable_headlight,
    disable_textures;
  BACKFACECULLINGMODE backface_culling;
  RGB 	outline_color,	/* color in which to draw outlines */
    bounding_box_color, /* color in which to draw bounding boxes */
    cluster_color;	/* color in which to show cluster bounding boxes */
  float   gamma;	/* gamma factor of display */
  float frames_per_sec;	/* statistics ... */
  int  animate;	        /* true for continuous animation */
} RENDEROPTIONS;

extern RENDEROPTIONS renderopts;

/* switches backface culling ... on when the argument is nonzero and off
 * if the argument is zero */
extern void RenderSetBackfaceCulling(BACKFACECULLINGMODE mode);
extern void RenderSetSmoothShading(char truefalse);
extern void RenderSetOutlineDrawing(char truefalse);
extern void RenderSetBoundingBoxDrawing(char truefalse);
extern void RenderSetClusterDrawing(char truefalse);
extern void RenderUseDisplayLists(char truefalse);

/* color for drawing outlines ... */
extern void RenderSetOutlineColor(RGB *outline_color);
extern void RenderSetBoundingBoxColor(RGB *outline_color);
extern void RenderSetClusterColor(RGB *outline_color);

/* sets current display gamma correction factor */
extern void RenderSetGamma(float gamma);

/* computes front- and backclipping plane distance for the current World and 
 * Camera */
extern void RenderGetNearFar(float *near, float *far);
extern void RenderSetBoundingBox(float *bounds);

#ifdef __cplusplus
}
#endif

#endif /*_RENDER_H_*/
