/* PhBPhongReflector.C: PhBPhongReflector nodes (public source) */

#include "PhBPhongReflector.H"
#include "matrix.H"

namespace xrml {

float PhBPhongReflector::glossy_threshold = 200;

void PhBPhongReflector::render(void)
{
  //
}

// returns approximate albedo of modified Phong BRDF: we assume that
// the cosine of the angle between outgoing direction and surface normal
// remains constant over the Phong lobe. This is a good approximation for
// sharp distributions.
float PhBPhongReflector::albedo(const Vec3& inDir,
				const complex& IndexIn,
				const complex& IndexOut,
				int modes)
{
  if (!(modes & SM_REFLECTION))
      return 0.;

  if ((sharpness == 0. && (modes & SM_DIFFUSE_REFLECTION)) ||
      (sharpness <= glossy_threshold && (modes & SM_GLOSSY_REFLECTION)) ||
      (sharpness > glossy_threshold && (modes & SM_SPECULAR_REFLECTION)))
    return fabs(inDir.z);
  else
    return 0.;
}

static Vec3 IdealReflectedDirection(const Vec3& inDir)
{
  return Vec3(-inDir.x, -inDir.y, inDir.z);
}

Vec3 PhBPhongReflector::sample(const Vec3& inDir,
			       const complex& IndexIn,
			       const complex& IndexOut,
			       float xi1, float xi2,
			       int modes)
{
  // rotation transforming the Z axis into
  // the ideal reflected direction.
  Vec3 Z = IdealReflectedDirection(inDir);
  double zz = sqrt(1 - Z.z*Z.z);
  Vec3 X = (zz < EPSILON)
    ? Vec3(1.,0.,0.)
    : Vec3(Z.y/zz, -Z.x/zz, 0.);
  Vec3 Y = Z ^ X;
  Mat3 rot(X,Y,Z);

  double phi = 2. * M_PI * xi1;
  double cos_phi = cos(phi);
  double sin_phi = sin(phi);
  double cos_theta = pow(xi2, 1./(sharpness+1.));
  double sin_theta = sqrt(1. - cos_theta*cos_theta);
  return Vec3(cos_phi * sin_theta,
	      sin_phi * sin_theta,
	      cos_theta) * rot;
}

float PhBPhongReflector::eval(const Vec3& inDir,
			      const Vec3& outDir,
			      const complex& IndexIn,
			      const complex& IndexOut,
			      int modes,
			      float *pdf)
{
  if (pdf) *pdf = 0.;

  if (!(modes & SM_REFLECTION) ||
      !((sharpness == 0. && (modes & SM_DIFFUSE_REFLECTION)) ||
	(sharpness <= glossy_threshold && (modes & SM_GLOSSY_REFLECTION)) ||
	(sharpness > glossy_threshold && (modes & SM_SPECULAR_REFLECTION)))) {
    // no contribution of the requested scattering mode.
    return 0.;
  }

  double cosa = IdealReflectedDirection(inDir) & outDir;
  if (cosa <= 0.)	// angle >= 90 degrees
    // pdf is 0: such outgoing rays cannot result from the sampling method
    // above.
    return 0.;

  double val = pow(cosa, sharpness);
  if (pdf) *pdf = val * (sharpness+1.) / (2. * M_PI);

  if ((outDir.z > 0.) != (inDir.z > 0.))
    // refracted direction
    // pdf is not 0: the sampling method above can yield rays
    // leaving the wrong side of the surface!
    return 0.;
  return val * (sharpness+2.) / (2. * M_PI);
}



}  // namespace xrml
