/* renderer.H: XRML node handlers */

#ifndef _RENDERER_H_
#define _RENDERER_H_

#include "xrml.H"
#include "matrix.H"
#include "world.H"
#include "StackManager.H"

namespace xrml {

class renderer {
public:
  virtual void begin_frame(class world *);
  virtual void end_frame(class world *);

  // WorldInfo
  virtual void worldInfo(class WorldInfo *);

  // bindables
  virtual void background(class Background *);
  virtual void fog(class Fog *);
  virtual void navigationInfo(class NavigationInfo *);
  virtual void viewpoint(class Viewpoint *);

  // grouping nodes
  virtual void begin_Grouping(class Grouping *);   // default grouping node handler
  virtual void end_Grouping(class Grouping *);
  virtual void begin_Anchor(class Anchor *);
  virtual void end_Anchor(class Anchor *);
  virtual void begin_Collision(class Collision *);
  virtual void end_Collision(class Collision *);
  virtual void begin_Group(class Group *);
  virtual void end_Group(class Group *);
  virtual void begin_Inline(class Inline *);
  virtual void end_Inline(class Inline *);
  virtual void begin_LOD(class LOD *);
  virtual void end_LOD(class LOD *);
  virtual void begin_Switch(class Switch *);
  virtual void end_Switch(class Switch *);

  // grouping nodes that transform their children (subclass of grouping nodes)
  virtual void begin_Transforming(class Transforming *); 	// default handler
  virtual void end_Transforming(class Transforming *);
  virtual void begin_Billboard(class Billboard *);
  virtual void end_Billboard(class Billboard *);
  virtual void begin_Transform(class Transform *);
  virtual void end_Transform(class Transform *);

  // light sources
  virtual void begin_LightSource(class LightSource *);
  virtual void end_LightSource(class LightSource *);
  virtual void begin_DirectionalLight(class DirectionalLight *);
  virtual void end_DirectionalLight(class DirectionalLight *);
  virtual void begin_PointLight(class PointLight *);
  virtual void end_PointLight(class PointLight *);
  virtual void begin_SpotLight(class SpotLight *);
  virtual void end_SpotLight(class SpotLight *);

  // shape
  virtual void begin_Shape(class Shape *);
  virtual void end_Shape(class Shape *);

  // appearance
  virtual void begin_Appearance(class Appearance *);
  virtual void end_Appearance(class Appearance *);

  // material
  virtual void begin_Material(class Material *);
  virtual void end_Material(class Material *);

  // texture transform
  virtual void begin_TextureTransform(class TextureTransform *);
  virtual void end_TextureTransform(class TextureTransform *);

  // texture nodes
  virtual void begin_Texture(class Texture *);	// default texture node handler
  virtual void end_Texture(class Texture *);
  virtual void begin_ImageTexture(class ImageTexture *);
  virtual void end_ImageTexture(class ImageTexture *);
  virtual void begin_MovieTexture(class MovieTexture *);
  virtual void end_MovieTexture(class MovieTexture *);
  virtual void begin_PixelTexture(class PixelTexture *);
  virtual void end_PixelTexture(class PixelTexture *);

  // geometry nodes
  // default geometry node handler for facetted geometry nodes
  // (not for PointSets or IndexedLineSets).
  virtual void geometry(class Geometry *);

  virtual void box(class Box *);
  virtual void cone(class Cone *);
  virtual void cylinder(class Cylinder *);
  virtual void elevationGrid(class ElevationGrid *);
  virtual void extrusion(class Extrusion *);
  virtual void indexedFaceSet(class IndexedFaceSet *);
  virtual void sphere(class Sphere *);
  virtual void text(class Text *);

  virtual void indexedLineSet(class IndexedLineSet *);
  virtual void pointSet(class PointSet *);

  // Sensors
  virtual void cylinderSensor(class CylinderSensor *);
  virtual void planeSensor(class PlaneSensor *);
  virtual void proximitySensor(class ProximitySensor *);
  virtual void sphereSensor(class SphereSensor *);
  virtual void timeSensor(class TimeSensor *);
  virtual void touchSensor(class TouchSensor *);
  virtual void visibilitySensor(class VisibilitySensor *);

  // Sound
  virtual void sound(class Sound *);


  //
  virtual void ignore(class SFNode *);

  // The default Geometry node handler Geometry(Geometry*)
  // - discretizes the geometry,
  // - computes default normals and texture coordinates if required and
  // - generates the following sequence of calls to renderer member functions
  // declared below:
  //
  // 1) begin_faces(Geometry *)
  //
  // 2) for each face:
  //	begin_face(int id, int nrverts)
  //	[face_normal(int id, SFVec3f)]		// if per face normals
  //	[face_color(int id, SFColor)]		// if per face colors
  //	for each vertex of the face:
  //		[vertex_normal(int id, SFVec3f)]// if per vertex normals
  //		[vertex_color(int id, SFColor)]	// if per vertex colors
  //		vertex_texCoord(int id, SFVec2f)
  //		vertex_coord(int id, SFVec3f)
  //	end_face(int id, int nverts)
  //
  // 3) end_faces(Geometry *)
  //
  virtual void begin_faces(class Geometry *);
  virtual void begin_face(int id, int nverts);
  virtual void face_normal(int id, const SFVec3f&);
  virtual void face_color(int id, const SFColor&);
  virtual void vertex_normal(int id, const SFVec3f&);
  virtual void vertex_color(int id, const SFColor&);
  virtual void vertex_texCoord(int id, const SFVec2f&);
  virtual void vertex_coord(int id, const SFVec3f&);
  virtual void end_face(int id, int nverts);
  virtual void end_faces(class Geometry *);
  // normal/texture coordinate/color generation can be disabled
  // by setting the following flags to false (default=true)
  bool normals_required, texcoords_required, colors_required;

  // Default PointSet renderer calls the following routines
  // in this order:
  // - begin_points()
  // - for each point:
  //   . [point_color()] if the point has a color associated
  //   . [point_normal()]                   normal
  //   . [point_texCoord()]                 texture coordinate
  //   . point_coord()
  // - end_points()
  virtual void begin_points(class PointSet*);
  virtual void point_color(const SFColor&);
  virtual void point_normal(const SFVec3f&);
  virtual void point_texCoord(const SFVec2f&);
  virtual void point_coord(const SFVec3f&);
  virtual void end_points(class PointSet*);

  // PhBRML node renderers:
  // appearance
  virtual void begin_PhBAppearance(class PhBAppearance *);
  virtual void end_PhBAppearance(class PhBAppearance *);

  // atmosphere
  virtual void phbAtmosphere(class PhBAtmosphere *);

  // background
  virtual void phbBackground(class PhBBackground *);	// default
  virtual void phbTexturedBackground(class PhBTexturedBackground *);
  virtual void phbProceduralBackground(class PhBProceduralBackground *);
  virtual void phbHomogeneousBackground(class PhBHomogeneousBackground *);
  virtual void phbBackgroundLight(class PhBBackgroundLight *);

  // monitor calibration (bindable)
  virtual void phbMonitorCalibration(class PhBMonitorCalibration *);
  
  // current transform and inverse, computed by the default Transforming node 
  // handler
  Mat4 xf, inverse_xf;

  // number of quart circle division used for discretizing spheres,
  // cylinders and cones.
  int nrQuartCircleDivisions;

  // constructor sets default values
  renderer() {
    nrQuartCircleDivisions = 5;
    normals_required = texcoords_required = colors_required = true;
  }

  virtual ~renderer() {}
};

} // namespace xrml

#endif /*_RENDERER_H_*/

