/* matrix.H: matrices */

#ifndef _XRML_MATRIX_H_
#define _XRML_MATRIX_H_

#include "vector.H"

namespace xrml {

struct Mat3: public PooledObj {
  float m[3][3];

  inline Mat3(const float x00 =1., const float x01 =0., const float x02 =0., 
       const float x10 =0., const float x11 =1., const float x12 =0.,
       const float x20 =0., const float x21 =0., const float x22 =1.) 
  {
    m[0][0] = x00; m[0][1] = x01; m[0][2] = x02;
    m[1][0] = x10; m[1][1] = x11; m[1][2] = x12;
    m[2][0] = x20; m[2][1] = x21; m[2][2] = x22;
  }
  
  // creates a transform to the coordinate system defined
  // by X, Y and Z
  inline Mat3(const Vec3& X, const Vec3& Y, const Vec3& Z)
  {
    m[0][0] = X.x; m[0][1] = X.y; m[0][2] = X.z;
    m[1][0] = Y.x; m[1][1] = Y.y; m[1][2] = Y.z;
    m[2][0] = Z.x; m[2][1] = Z.y; m[2][2] = Z.z;
  }

  // creates a rotation transform
  Mat3(const Vec4& sr)
  {
    float x=sr.x, y=sr.y, z=sr.z, a=sr.w, c, s, t, A, B, C, D, E, F;
    s = sqrt(x*x + y*y + z*z);
    x /= s; y /= s; z /= s;
    c = cos(a); s = sin(a); t = 1-c;
    A = t*x*y; B = s*z;
    C = t*x*z; D = s*y;
    E = t*y*z; F = s*x;

    m[0][0] = t*x*x + c; m[0][1] = A+B; m[0][2] = C-D;
    m[1][0] = A-B; m[1][1] = t*y*y + c; m[1][2] = E+F;
    m[2][0] = C+D; m[2][1] = E-F; m[2][2] = t*z*z + c;
  }

  // creates a scaling transform
  inline Mat3(const Vec3& s) 
  {
    m[0][0] = s.x; m[0][1] = m[0][2] = 0.;
    m[1][1] = s.y; m[1][0] = m[1][2] = 0.;	
    m[2][2] = s.z; m[2][0] = m[2][1] = 0.;
  }

  // extracts the upper 3x3 matrix of the transform
  Mat3(const class Mat4& v);

  inline const float* operator[](const int i) const { return m[i]; }

  // Vec3 = Mat3 * Vec3: multiply with transpose.
  inline const Vec3 operator*(const Vec3& v) const
  {
    return Vec3(m[0][0] * v.x + m[0][1] * v.y + m[0][2] * v.z,
		m[1][0] * v.x + m[1][1] * v.y + m[1][2] * v.z,
		m[2][0] * v.x + m[2][1] * v.y + m[2][2] * v.z);
  }

  // returns transpose
  inline const Mat3 transpose(void) const
  {
    return Mat3(m[0][0], m[1][0], m[2][0],
		m[0][1], m[1][1], m[2][1],
		m[0][2], m[1][2], m[2][2]);
  }

  inline const Mat3 operator*(const Mat3& t) const
  {
    return Mat3(m[0][0] * t.m[0][0] + m[0][1] * t.m[1][0] + m[0][2] * t.m[2][0],
		m[0][0] * t.m[0][1] + m[0][1] * t.m[1][1] + m[0][2] * t.m[2][1],
		m[0][0] * t.m[0][2] + m[0][1] * t.m[1][2] + m[0][2] * t.m[2][2],
		//
		m[1][0] * t.m[0][0] + m[1][1] * t.m[1][0] + m[1][2] * t.m[2][0],
		m[1][0] * t.m[0][1] + m[1][1] * t.m[1][1] + m[1][2] * t.m[2][1],
		m[1][0] * t.m[0][2] + m[1][1] * t.m[1][2] + m[1][2] * t.m[2][2],
		//
		m[2][0] * t.m[0][0] + m[2][1] * t.m[1][0] + m[2][2] * t.m[2][0],
		m[2][0] * t.m[0][1] + m[2][1] * t.m[1][1] + m[2][2] * t.m[2][1],
		m[2][0] * t.m[0][2] + m[2][1] * t.m[1][2] + m[2][2] * t.m[2][2]);
  }

  inline const Mat3& operator*=(const Mat3& t)
  {
    return (*this) = (*this) * t;
  }

  friend ostream& operator<<(ostream& s, const Mat3& v)
  {
    return s << v.m[0][0] << ' ' << v.m[0][1] << ' ' << v.m[0][2] << '\n'
	     << v.m[1][0] << ' ' << v.m[1][1] << ' ' << v.m[1][2] << '\n'
	     << v.m[2][0] << ' ' << v.m[2][1] << ' ' << v.m[2][2] << '\n';
  }
};

struct Mat4: public PooledObj  {
  float m[4][4];

  inline Mat4(const float x00 =1., const float x01 =0., const float x02 =0., 
	      const float x10 =0., const float x11 =1., const float x12 =0.,
	      const float x20 =0., const float x21 =0., const float x22 =1.) 
  {
    m[0][0] = x00; m[0][1] = x01; m[0][2] = x02; m[0][3] = 0.;
    m[1][0] = x10; m[1][1] = x11; m[1][2] = x12; m[1][3] = 0.;
    m[2][0] = x20; m[2][1] = x21; m[2][2] = x22; m[2][3] = 0.;
    m[3][0] = 0. ; m[3][1] = 0. ; m[3][2] = 0. ; m[3][3] = 1.;
  }

  inline Mat4(const Mat3& t) {
    m[0][0] = t.m[0][0]; m[0][1] = t.m[0][1]; m[0][2] = t.m[0][2]; m[0][3] = 0.;
    m[1][0] = t.m[1][0]; m[1][1] = t.m[1][1]; m[1][2] = t.m[1][2]; m[1][3] = 0.;
    m[2][0] = t.m[2][0]; m[2][1] = t.m[2][1]; m[2][2] = t.m[2][2]; m[2][3] = 0.;
    m[3][0] = 0. ; m[3][1] = 0. ; m[3][2] = 0. ; m[3][3] = 1.;
  }

  inline Mat4(const float x00, const float x01, const float x02, const float x03, 
	      const float x10, const float x11, const float x12, const float x13,
	      const float x20, const float x21, const float x22, const float x23,
	      const float x30 =0., const float x31 =0., const float x32 =0., const float x33 =1.)
  {
    m[0][0] = x00; m[0][1] = x01; m[0][2] = x02; m[0][3] = x03;
    m[1][0] = x10; m[1][1] = x11; m[1][2] = x12; m[1][3] = x13;
    m[2][0] = x20; m[2][1] = x21; m[2][2] = x22; m[2][3] = x23;
    m[3][0] = x30; m[3][1] = x31; m[3][2] = x32; m[3][3] = x33;
  }

  Mat4 const operator*(const Mat4& t) const
  {
    return Mat4(
    m[0][0]*t.m[0][0] + m[0][1] * t.m[1][0] + m[0][2] * t.m[2][0] + m[0][3] * t.m[3][0],
    m[0][0]*t.m[0][1] + m[0][1] * t.m[1][1] + m[0][2] * t.m[2][1] + m[0][3] * t.m[3][1],
    m[0][0]*t.m[0][2] + m[0][1] * t.m[1][2] + m[0][2] * t.m[2][2] + m[0][3] * t.m[3][2],
    m[0][0]*t.m[0][3] + m[0][1] * t.m[1][3] + m[0][2] * t.m[2][3] + m[0][3] * t.m[3][3],
    //
    m[1][0]*t.m[0][0] + m[1][1] * t.m[1][0] + m[1][2] * t.m[2][0] + m[1][3] * t.m[3][0],
    m[1][0]*t.m[0][1] + m[1][1] * t.m[1][1] + m[1][2] * t.m[2][1] + m[1][3] * t.m[3][1],
    m[1][0]*t.m[0][2] + m[1][1] * t.m[1][2] + m[1][2] * t.m[2][2] + m[1][3] * t.m[3][2],
    m[1][0]*t.m[0][3] + m[1][1] * t.m[1][3] + m[1][2] * t.m[2][3] + m[1][3] * t.m[3][3],
    //
    m[2][0]*t.m[0][0] + m[2][1] * t.m[1][0] + m[2][2] * t.m[2][0] + m[2][3] * t.m[3][0],
    m[2][0]*t.m[0][1] + m[2][1] * t.m[1][1] + m[2][2] * t.m[2][1] + m[2][3] * t.m[3][1],
    m[2][0]*t.m[0][2] + m[2][1] * t.m[1][2] + m[2][2] * t.m[2][2] + m[2][3] * t.m[3][2],
    m[2][0]*t.m[0][3] + m[2][1] * t.m[1][3] + m[2][2] * t.m[2][3] + m[2][3] * t.m[3][3],
    //
    m[3][0]*t.m[0][0] + m[3][1] * t.m[1][0] + m[3][2] * t.m[2][0] + m[3][3] * t.m[3][0],
    m[3][0]*t.m[0][1] + m[3][1] * t.m[1][1] + m[3][2] * t.m[2][1] + m[3][3] * t.m[3][1],
    m[3][0]*t.m[0][2] + m[3][1] * t.m[1][2] + m[3][2] * t.m[2][2] + m[3][3] * t.m[3][2],
    m[3][0]*t.m[0][3] + m[3][1] * t.m[1][3] + m[3][2] * t.m[2][3] + m[3][3] * t.m[3][3]);
  }

  inline const Mat4& operator*=(const Mat4& t)
  {
    return (*this) = (*this) * t;
  }

  inline const Mat4 rotation(const Vec4& rot) const
  {
    return Mat4(Mat3(rot));
  }

  inline const Mat4 rotate(const Vec4& rot)
  {
    return (*this) *= rotation(rot);
  }

  inline const Mat4 scaling(const Vec3& s) const
  {
    return Mat4(s.x, 0, 0, 0,
		0, s.y, 0, 0, 
		0, 0, s.z, 0,
		0, 0, 0,   1);
  }

  inline const Mat4 scale(const Vec3& s)
  {
    return (*this) *= scaling(s);
  }

  inline const Mat4 translation(const Vec3& t) const
  {
    return Mat4(1  , 0  , 0  , 0,
		0  , 1  , 0  , 0, 
		0  , 0  , 1  , 0,
		t.x, t.y, t.z, 1);
  }

  inline const Vec3 get_translation(void) const
  {
    return Vec3(m[3][0], m[3][1], m[3][2]);
  }

  inline const Mat4 translate(const Vec3& t) 
  {
    return (*this) *= translation(t);
  }

  inline const float* operator[](const int i) const { return m[i]; }

  inline const Vec3 operator*(const Vec3& v) const
  {
    double w = m[3][0] * v.x + m[3][1] * v.y + m[3][2] * v.z + m[3][3];
    return Vec3((m[0][0] * v.x + m[0][1] * v.y + m[0][2] * v.z + m[0][3])/w,
		(m[1][0] * v.x + m[1][1] * v.y + m[1][2] * v.z + m[1][3])/w,
		(m[2][0] * v.x + m[2][1] * v.y + m[2][2] * v.z + m[2][3])/w);
  }

  inline const Vec4 operator*(const Vec4& v) const
  {
    return Vec4(m[0][0] * v.x + m[0][1] * v.y + m[0][2] * v.z + m[0][3] * v.w,
		m[1][0] * v.x + m[1][1] * v.y + m[1][2] * v.z + m[1][3] * v.w,
		m[2][0] * v.x + m[2][1] * v.y + m[2][2] * v.z + m[2][3] * v.w,
		m[3][0] * v.x + m[3][1] * v.y + m[3][2] * v.z + m[3][3] * v.w);
  }

  // returns transpose
  inline const Mat4 transpose(void) const
  {
    return Mat4(m[0][0], m[1][0], m[2][0], m[3][0],
		m[0][1], m[1][1], m[2][1], m[3][1],
		m[0][2], m[1][2], m[2][2], m[3][2],
		m[0][3], m[1][3], m[2][3], m[3][3]);
  }

  friend ostream& operator<<(ostream& s, const Mat4& v)
  {
    return s << v.m[0][0] << ' ' << v.m[0][1] << ' ' << v.m[0][2] << ' ' << v.m[0][3] << '\n'
	     << v.m[1][0] << ' ' << v.m[1][1] << ' ' << v.m[1][2] << ' ' << v.m[1][3] << '\n'
	     << v.m[2][0] << ' ' << v.m[2][1] << ' ' << v.m[2][2] << ' ' << v.m[2][3] << '\n'
	     << v.m[3][0] << ' ' << v.m[3][1] << ' ' << v.m[3][2] << ' ' << v.m[3][3] << '\n';
  }
};

inline Mat3::Mat3(const Mat4& v)
{
  m[0][0] = v.m[0][0]; m[0][1] = v.m[0][1]; m[0][2] = v.m[0][2]; 
  m[1][0] = v.m[1][0]; m[1][1] = v.m[1][1]; m[1][2] = v.m[1][2]; 
  m[2][0] = v.m[2][0]; m[2][1] = v.m[2][1]; m[2][2] = v.m[2][2]; 
}

inline const Vec3 Vec3::operator*(const Mat3& t) const
{
  return Vec3(x * t.m[0][0] + y * t.m[1][0] + z * t.m[2][0],
	      x * t.m[0][1] + y * t.m[1][1] + z * t.m[2][1],
	      x * t.m[0][2] + y * t.m[1][2] + z * t.m[2][2]);
}

inline const Vec3& Vec3::operator*=(const Mat3& t)
{
  Vec3 a(*this);
  x = a.x * t.m[0][0] + a.y * t.m[1][0] + a.z * t.m[2][0];
  y = a.x * t.m[0][1] + a.y * t.m[1][1] + a.z * t.m[2][1];
  z = a.x * t.m[0][2] + a.y * t.m[1][2] + a.z * t.m[2][2];
  return *this;
}

inline const Vec3 Vec3::operator*(const Mat4& t) const
{
  double w = x * t.m[0][3] + y * t.m[1][3] + z * t.m[2][3] + t.m[3][3];
  return Vec3((x * t.m[0][0] + y * t.m[1][0] + z * t.m[2][0] + t.m[3][0])/w,
	      (x * t.m[0][1] + y * t.m[1][1] + z * t.m[2][1] + t.m[3][1])/w,
	      (x * t.m[0][2] + y * t.m[1][2] + z * t.m[2][2] + t.m[3][2])/w);
}

inline const Vec3& Vec3::operator*=(const Mat4& t)
{
  double w = x * t.m[0][3] + y * t.m[1][3] + z * t.m[2][3] + t.m[3][3];
  Vec3 a(*this);
  x = (a.x * t.m[0][0] + a.y * t.m[1][0] + a.z * t.m[2][0] + t.m[3][0])/w;
  y = (a.x * t.m[0][1] + a.y * t.m[1][1] + a.z * t.m[2][1] + t.m[3][1])/w;
  z = (a.x * t.m[0][2] + a.y * t.m[1][2] + a.z * t.m[2][2] + t.m[3][2])/w;
  return *this;
}

inline const Vec4 Vec4::operator*(const Mat4& t) const
{
  return Vec4(x * t.m[0][0] + y * t.m[1][0] + z * t.m[2][0] + w * t.m[3][0],
	      x * t.m[0][1] + y * t.m[1][1] + z * t.m[2][1] + w * t.m[3][1],
	      x * t.m[0][2] + y * t.m[1][2] + z * t.m[2][2] + w * t.m[3][2],
	      x * t.m[0][3] + y * t.m[1][3] + z * t.m[2][3] + w * t.m[3][3]);
}

inline const Vec4& Vec4::operator*=(const Mat4& t)
{
  Vec4 a(*this);
  x = a.x * t.m[0][0] + a.y * t.m[1][0] + a.z * t.m[2][0] + a.w * t.m[3][0];
  y = a.x * t.m[0][1] + a.y * t.m[1][1] + a.z * t.m[2][1] + a.w * t.m[3][1];
  z = a.x * t.m[0][2] + a.y * t.m[1][2] + a.z * t.m[2][2] + a.w * t.m[3][2];
  w = a.x * t.m[0][3] + a.y * t.m[1][3] + a.z * t.m[2][3] + a.w * t.m[3][3];
  return *this;
}

}  // namespace xrml

#endif /*_XRML_MATRIX_H_*/
