;; Wojciech Jedynak (wjedynak@gmail.com)

;; Part 03: Booleans -- comparison between deterministic reductions
;; and the unrestricted calculus

;;#lang racket
(require redex)

;; 1) language definition -- common base

(define-language B
  (e true
     false
     (or e e)))

;; 1a) CBN

(define-extended-language BN B
  (E hole
     (or E e)))

;; 1b) left-to-right CBV

(define-extended-language BV B
  (E hole
     (or E e)
     (or v E))
  (v true
     false))

;; 1c) unrestricted beta

(define-extended-language BB B
  (E hole
     (or E e)
     (or e E)))

;; 3) A reduction relation

(define redn
  (reduction-relation
   BN
   (--> (in-hole E (or true e))
        (in-hole E true)
        "or-true")

   (--> (in-hole E (or false e))
        (in-hole E e)
        "or-false")))

(define redv
  (reduction-relation
   BV
   (--> (in-hole E (or true v))
        (in-hole E true)
        "or-true")

   (--> (in-hole E (or false v))
        (in-hole E v)
        "or-false")))

(define redb
  (reduction-relation
   BB
   (--> (in-hole E (or true e))
        (in-hole E true)
        "or-true")

   (--> (in-hole E (or false e))
        (in-hole E e)
        "or-false")))


;; 4a) The reduction relation in action

(traces redn (term (or true false)))

(traces redn (term (or (or true false)
                      (or false true))))

(traces redn (term (or (or (or false true) false)
                      (or false true))))

;; 4b) The reduction relation in action

(traces redv (term (or true false)))

(traces redv (term (or (or true false)
                      (or false true))))

(traces redv (term (or (or (or false true) false)
                      (or false true))))

;; 5) How do we check some properties of the relations?

(apply-reduction-relation redv (term true))
(apply-reduction-relation redv (term (or true false)))

;; 5a) is the relation completely deterministic?
;;     i.e. is there at most one successor state?

(define (deterministic? rel argument)
  (let [(result (apply-reduction-relation rel argument))]
    (match result
      ['() #t]
      [(list a) (deterministic? rel a)]
      [else #f])))

;; CBN

(deterministic? redv (term (or true false)))

(redex-check B e (deterministic? redn (term e)))

(redex-check B e (begin
                   (printf "~s\n" (term e))
                   (deterministic? redn (term e))))

(redex-check BN e (begin
                   (printf "~s\n" (term e))
                   (deterministic? redn (term e)))
             #:source redn
             #:attempts 20)

;; CBV

(redex-check B e (deterministic? redv (term e)))

(redex-check BV e (begin
                   (printf "~s\n" (term e))
                   (deterministic? redv (term e)))
             #:source redn
             #:attempts 200)

;; 5b) CBN is equivalent to CBV (result-wise)

(define (equivalent? e)
  (equal? (apply-reduction-relation* redv e)
          (apply-reduction-relation* redn e)))

(redex-check B e (equivalent? (term e)))

;; 5c) efficiency comparison

(define (how-many-times???? rel e n)
  (let [(result (apply-reduction-relation rel e))]
    (match result
      ['() n]
      [else (apply min (map (lambda (e1) (how-many-times???? rel e1 (+ 1 n))) result))])))

(how-many-times???? redv (term (or true (or true false))) 0)
(how-many-times???? redn (term (or true (or true false))) 0)
(how-many-times???? redb (term (or true (or true false))) 0)

;; conjecture: cbn is the best

(redex-check B e (= (how-many-times???? redb (term e) 0)
                    (how-many-times???? redn (term e) 0)))
