;; Wojciech Jedynak (wjedynak@gmail.com)

;; Part 04: Generalized Hutton's Razor -- simple arithmetic expressions

;; #lang racket
(require redex)

;; Language definition

(define-language L
  (e
   n
   (+ e ...))
  (n number))

;; examples matchings

(redex-match
 L
 e
 123)

;; matching of subterms
(redex-match L (+ e_1 e_2 e_3 ...) (term (+ 1 2 3 4)))

;; match a list
(redex-match L (e_0 e_1 ... e e_3 ...) (term (1 2 3 4 5)))

;; patterns can be non-linear
(redex-match L (e e) (term (1 1)))

;; we can also name the length of a variable-pattern!
(redex-match L (e ..._1 e_0 e_2 ..._1) (term (1 2 3 4 5)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; reduction relation

(define-extended-language Ev L
  (E hole
     (+ e ... E e ...)))

;; We need to define a metafunction that will actually add the numbers

(define-metafunction Ev
  sum : n ... -> n
  [(sum n ...)
   ,(apply + (term (n ...)))])

(define red
  (reduction-relation
   Ev
   (--> (in-hole E (+   n ...))
        (in-hole E (sum n ...))
        "plus")))

;; tracing: the reduction graph

(traces red (term (+ 1 2)))
(traces red (term (+ (+ 1 2) 3)))
(traces red (term (+ 3 (+ 1 2))))

(define square
  (term (+ (+ 1 2) (+ 3 4))))
(traces red square)

(define cube
  (term (+ (+ 1 2) (+ 3 4) (+ 5 6))))
(traces red cube)

;; interactive stepper

(stepper red cube)

;; unit tests

(test-->>                               ; transitive closure
 red
 square
 (term 10))

(test-->>
 red
 cube
 (term 21))

(test-results)

;; all results must be listed!

(test-->
 red
 square
 (term (+ 3 (+ 3 4)))
 (term (+ (+ 1 2) 7)))

(test-results)

;; apply-reduction-relation

(apply-reduction-relation red (term 1))
(apply-reduction-relation red square)

(length (apply-reduction-relation* red square))

;; rendering

(render-reduction-relation red)

;; random testing

(define (single-result? e)
  (>= 1 (length (apply-reduction-relation red e))))

(redex-check Ev e (single-result? (term e))
             #:attempts 100)
