(* shift-reset.sml *)
(* SML code from
     Representing Monads
   by A. Filinski at POPL 1994
   as reproduced in Appendix C of
     A Dynamic Continuation-Passing Style for Dynamic Delimited Continuations
   by D. Biernacki, O. Danvy, K. Millikin
   Version of 7 December 2014
*)

(* ********** *)

signature ESCAPE
= sig
    type void
    val coerce : void -> 'a
    val escape : (('a -> void) -> 'a) -> 'a
  end;

structure Escape : ESCAPE
= struct
    open SMLofNJ.Cont
    datatype void = VOID of void
    fun coerce (VOID v) = coerce v
    fun escape f = callcc (fn k => f (fn x => throw k x))
  end

signature SHIFT_AND_RESET
= sig
    type answer
    val shift : (('a -> answer) -> answer) -> 'a
    val reset : (unit -> answer) -> answer
    val run : (unit -> answer) -> answer
  end;

functor make_Shift_and_Reset (type answer) : SHIFT_AND_RESET
= struct
    open Escape
    type answer = answer
    exception MISSING_RESET
    val mk : (answer -> void) ref = ref (fn _ => raise MISSING_RESET)
    fun abort x = coerce (!mk x)
    fun reset t
        = escape (fn k => let val m = !mk
                          in mk := (fn r => (mk := m; k r));
                             abort (t ())
                          end)
    fun shift h
        = escape (fn k => abort (h (fn v => reset (fn () => coerce (k v)))))
    fun run t
        = reset t
  end;

(* ********** *)

(* end of shift-reset.sml *)
