(** Main file: proof of weak head normalization in system T, 
     for closed terms

     M. Biernacka, June 15, 2007    
*)

Require Export Typing. 

Set Implicit Arguments.

(* Redefinition of pairs on [Type] instead of [Set] *)

Inductive prod (A B:Type) : Type :=
    pair : A -> B -> prod A B.
Notation "x * y" := (prod x y) : type_scope.

Definition fst (A B:Type)(ab:prod A B) := let (a,_) := ab in a.
Definition snd (A B:Type)(ab:prod A B) := let (_,b) := ab in b.

Inductive halfprod (A:Prop)(B:Type) : Type := 
   halfpair : A -> B -> halfprod A B. 
Notation "x ** y" := (halfprod x y) (at level 40) : type_scope.

Open Scope type_scope.

Module Type RedSem.

(* one-step reduction *)
 Parameter Rd : type -> term -> term -> Prop.
 
(* refl-transitive closure of Rd *)
 Inductive RdTrans : type -> term -> term -> Prop :=
 | rd_refl : forall rho r, RdTrans rho r r
 | rd_trans : forall rho r s t, Rd rho r s -> 
   RdTrans rho s t ->
   RdTrans rho r t.	
 
(* syntactic characterization of values *)
 Inductive Val : type -> term -> Prop :=
 | val_zero : Val Int Zero
 | val_succ : forall r, Val Int r -> Val Int (^r)
 | val_lam : forall rho r sigma, Val (rho-->sigma) (\rho, r).

(* evaluation *)
 Definition Ev (rho:type) (r s:term) : Prop :=
   (Val rho s) /\ RdTrans rho r s.

(* required properties of red. semantics *)

 Axiom ax_beta : 
   forall r s rho sigma rs,
     Rd sigma ((sub (\rho,r) rs);s) (sub r ((s::rs)#rs.(shift))).

 Axiom ax_ctx_app :
   forall r s t rho sigma, 
     Rd (rho-->sigma) r s -> Rd sigma (r;t) (s;t).

 Axiom ax_refl_trans_clo :
   forall r s t rho, 
     Rd rho r s -> Ev rho s t -> Ev rho r t.
 
 Axiom ax_lam_val : 
   forall r rho sigma,	 
     Ev (rho --> sigma) (\rho, r) (\rho, r). 	

 Axiom ax_zero_val : 
    Ev Int Zero Zero.

 Axiom ax_ev_succ : 
    forall r s,
      Ev Int r s -> Ev Int (^r) (^s).

 Axiom ax_ctx_r :
   forall rho r r' s t, 
     Rd Int r r' -> Rd rho (R rho r s t) (R rho r' s t).

 Axiom ax_zero_r :
   forall rho r s,
     Rd rho (R rho Zero r s) r.

 Axiom ax_succ_r :
   forall rho r s t,
     Rd rho (R rho (^r) s t) ((t;r);R rho r s t).  
  
End RedSem.

Module NormalizationProof (R:RedSem).

Import R.

(* termination with a value *)
Definition halts (rho:type) (r:term) :=
  { v: term | Ev rho r v }.

(* Tait's (weak) computability predicate *)
Fixpoint RR (rho:type) (r:term) {struct rho} : Type :=
  (TypJ nil r rho) **
  match rho with
  | Int => halts rho r
  | rho-->sigma => halts (rho-->sigma) r *
    (forall s:term, RR rho s -> RR sigma (r;s))
  end.  	

(* list of terms with RR property *)
Fixpoint RRs (rhos : context) (rs : list term) {struct rhos} : Type :=
  match rhos, rs with
  | nil, nil => True
  | nil, _ => False
  | _, nil => False
  | (rho :: rhos), (r :: rs) => 
    prod (RR rho r) (RRs rhos rs)
  end.	


Lemma RRs_nth : 
  forall (rhos : list type) (rho:type) (rs:  list term) (r:term) n, 
    n < length rhos -> RRs rhos rs -> RR (nth n rhos rho) (nth n rs r).

Proof.
induction rhos; destruct rs.
 intros.
   elimtype False.
   simpl in H.
   intuition.
 intros.
   simpl in X.
   destruct X.
 destruct n.
  intros.
    simpl in |- *.
    firstorder.
  intros.
    simpl in |- *.
    firstorder.
 intros.
   inversion X.
   elim X.
   intros.
   destruct n.
  simpl in |- *.
    auto.
  simpl in |- *.
    apply IHrhos.
   auto with arith.
   auto.
Qed.
    
(* some inversion lemmas *)

(* if term has the property, then it halts *)

Lemma RR_implies_halt : 
  forall (rho:type) (r:term), 
    TypJ nil r rho -> RR rho r -> halts rho r.

Proof.
 induction rho; intros; inversion X; auto.
 destruct X0; auto. 
Qed.
 
(* if term has the property, then it is well-typed *)
Lemma RR_implies_typcor :
  forall (rho:type) (r:term), 
    RR rho r -> TypJ nil r rho.

Proof.
intros.
induction rho.
 simpl in X.
   elim X.
   trivial.
 elim X.
   trivial.
Qed.

(* one-step reduction preserves RR backwards *)

Lemma Red_inv_preserves_RR : 
  forall (rho:type) (r s:term),
    TypJ nil r rho -> Rd rho r s -> RR rho s -> RR rho r.

Proof.
induction rho.
 intros.
   inversion X.
   elim H2.
   intros.
   simpl in |- *; split; trivial.
   exists x.
   apply (ax_refl_trans_clo H0 p).

 intros.
  split.
  trivial.

 intros.
   split; auto.
   destruct X; auto.
   fold RR in y.
   destruct y.
   destruct h.
   split with x; auto.
   apply ax_refl_trans_clo with s; auto.
   
intros.
    assert (Rd rho2 (r; s0) (s; s0)).
    apply (ax_ctx_app s0 H0).
    elim X.
    intros.
    destruct b.
    assert (RR rho2 (s; s0)).
     apply (r0 s0 X0).
    trivial.
    apply IHrho2 with (r := r; s0) (s := s; s0); auto.
    apply (TypJ_App nil rho1 rho2 r s0); trivial.
    apply (RR_implies_typcor rho1 s0 X0); trivial.
Qed.

(* some transitivity lemmas *)

Lemma Trans : 
  forall r s t rho, 
    RdTrans rho r s -> RdTrans rho s t -> RdTrans rho r t.

Proof. 
induction 1 using RdTrans_ind; intros; auto.
constructor 2 with s; auto.
Qed.

Lemma TransApp : 
  forall r s t rho sigma,
    TypJ nil t rho -> RdTrans (rho-->sigma) r s -> RdTrans sigma (r;t) (s;t).

Proof.
set (Q:= fun rho r s => 
  match rho with
    Int => True
  | rho0-->rho1 => forall t, RdTrans rho1 (r;t) (s;t)
  end).
assert (IndHyp : forall rho r s, RdTrans rho r s -> Q rho r s).
Focus 2.
intros; unfold Q in IndHyp.
assert (hh:=IndHyp (rho-->sigma) r s H0).
simpl in hh.
apply hh; auto.

unfold Q; intros; auto.
induction H.
induction rho; auto.
intros; constructor.
induction rho; auto.
intros.
constructor 2 with (s;t0); auto.
apply ax_ctx_app with rho1; auto.
Qed.

Lemma RedTrans_inv_preserves_RR :
  forall (rho:type) (r s:term),
    TypJ nil r rho -> RdTrans rho r s -> RR rho s -> RR rho r.  
Proof.
induction rho.
 intros.
   simpl in |- *.
   split.
   auto.
  inversion X.
    inversion H2.
    exists x.
    split; auto.
    inversion H3; auto.
    apply Trans with (s := s); auto.
    inversion H3; auto.
 
 intros.
   simpl in |- *.
   split.
   auto.
  intros.
   destruct X.
   fold RR in y.
   destruct y.
  split; auto.
  destruct h.
  split with x; auto.
  destruct e.
  split; auto.
  apply Trans with s; auto.
 intros.  
  assert (RR rho2 (s;s0)).
  apply r0; auto.
  apply IHrho2 with (s;s0); auto.
 apply TypJ_App with rho1; auto.
  apply RR_implies_typcor; auto.
  apply TransApp with rho1; auto.
  apply RR_implies_typcor; auto.
Qed.

Lemma RRs_length : forall (rhos: list type) (rs: list term),
  RRs rhos rs -> length rhos = length rs.

Proof.
induction rhos.
 destruct rs.
  auto.
  simpl in |- *.
    intro.
    elim H.
 destruct rs.
  simpl in |- *.
    intuition.
  intros.
    simpl in |- *.
    assert (length rhos = length rs).
   apply IHrhos.
     simpl in X.
     elim X.
     trivial.
   auto with arith.
Qed.

Lemma RdTrans_ctx_r :
  forall rho r u,
     RdTrans rho r u -> 
       match rho with
       | Int => forall sigma s t, RdTrans sigma (R sigma r s t) (R sigma u s t)
       | _ => True
       end.

Proof.
intros.
apply RdTrans_ind with 
(P:= fun (rho:type) r u => 
  match rho with
  | Int => forall sigma s t, RdTrans sigma (R sigma r s t) (R sigma u s t)
  | _ => True
  end); auto; intros.
induction rho0; auto.
intros.
constructor.
induction rho0; auto.
intros.
constructor 2 with (R sigma s s0 t0); auto.
apply ax_ctx_r; trivial.
Qed.

Lemma val_int_wt :
  forall v, 
    Val Int v -> TypJ nil v Int.

Proof.
induction v; intros; auto; inversion H; subst.
split; auto.
constructor; auto; elim (IHv H1); auto.
Qed.

Lemma R_aux :
  forall v rho r s t,
    RdTrans Int r v -> Val Int v ->
    TypJ nil (R rho r s t) rho -> RR rho s -> RR (Int-->rho-->rho) t ->
    RR rho (R rho r s t).

Proof.
induction v; intros; try (elimtype False; inversion H0; subst; contradiction; fail);
assert (hh:=RdTrans_ctx_r H); simpl in hh.
assert (RdTrans rho (R rho r s t) s).
apply Trans with (R rho Zero s t).
apply hh; auto.
constructor 2 with s; auto.
apply ax_zero_r; auto.
constructor.
apply RedTrans_inv_preserves_RR with s; auto.
assert (RdTrans rho (R rho r s t) ((t;v);R rho v s t)); auto.
apply Trans with (R rho (^v) s t).
apply hh; auto.
constructor 2 with ((t;v);R rho v s t); auto.
apply ax_succ_r; auto.
constructor.
apply RedTrans_inv_preserves_RR with ((t;v);R rho v s t); auto.

inversion X0; subst; auto.
fold RR in X1.
destruct X1.
elim (h0 v); auto.
intros.
destruct b.
apply r0; auto.
apply IHv; auto.
constructor.
inversion H0; auto.
split; auto.
destruct H1.
inversion H1; subst; repeat constructor; auto.
inversion H0; subst; auto.
elim (val_int_wt H6); auto.
inversion H0; subst.
elim (val_int_wt H6); auto.
repeat split; auto. 
inversion H0; subst; elim (val_int_wt H5); auto.
inversion H0; subst; elim (val_int_wt H5); auto.
split with v; auto.
split; auto.
inversion H0; auto.
constructor.
Qed.

(* main lemma - RR is preserved by substitution *)

Lemma subst_preserves_RR : 
  forall (r: term) (rs: substitution) (rhos:context)(rho:type),
    RRs rhos rs -> TypJ rhos r rho -> RR rho (sub r rs). 

Proof.
induction r.
 intros; simpl; auto.
 inversion_clear H.
 simpl in H1; subst.
 simpl.
 split; auto.
 split with Zero; apply ax_zero_val. 
(* succ *)
 intros; simpl; inversion_clear H; simpl; auto.
 simpl in H1.
 subst rho.
 simpl;  split; auto.
 inversion_clear H0.
 assert (RR Int (sub r rs)).
 apply IHr with rhos; auto.
 split; auto.
 constructor; auto.
 assert (TypJ nil (sub r rs) Int).
 apply RR_implies_typcor; auto.
 inversion_clear H0; auto.
 assert (TypJ nil (sub r rs) Int).
 apply RR_implies_typcor; auto.
 inversion_clear H0; auto.
 assert (RR Int (sub r rs)).
 apply IHr with rhos; auto.
 inversion_clear H0.
 split; auto. 
 assert (TypJ nil (sub r rs) Int).
 apply RR_implies_typcor; auto.
 assert (hh:=RR_implies_halt H X0).
 destruct hh.
 split with (^x); auto.
 apply ax_ev_succ; auto. 
(* Var *)
 intros.
   elim H.
   intros.
   rewrite <- H1.
   simpl in |- *.
   apply RRs_nth.
  inversion H0.
    auto.
  auto.
(* App *)
 intros.
   simpl in |- *.
   set (sigma := typ rhos r2) in |- *.
   assert (H1 : TypJ rhos r1 (sigma --> rho)).
  eauto.
  assert (H2 : TypJ rhos r2 sigma).
   eauto.
  assert (RR sigma (sub r2 rs)).
    apply IHr2 with (rhos := rhos).
     trivial.
     trivial.
    assert (RR (sigma --> rho) (sub r1 rs)).
     apply IHr1 with (rhos := rhos).
      trivial.
      trivial.
     inversion H1.
       elim X1.
       intros.
       fold RR in b.
       fold RR in a.
       fold RR in X0.
       destruct b.
       apply (r (sub r2 rs)).
       trivial.
(* Abs *)
 intros.
   destruct H.
   subst rho.
   assert (TypJ nil (sub (\ t, r) rs) (t --> typ (t :: rhos) r)).
  apply TypJ_sub2 with rhos.
   assert (length rhos = length rs).
    apply RRs_length.
     intuition.
    intuition.
   apply TypJ_Abs with (mu := typ (t :: rhos) r).
    trivial.
    unfold TypJ in |- *.
      split.
     inversion H.
       trivial.
     trivial.
   intros.
     assert (RR (nth n rhos d') (nth n rs d)).
    apply RRs_nth.
     trivial.
     trivial.
    apply RR_implies_typcor.
      trivial.
  simpl in |- *.
    split.
   simpl in H0.
     trivial.
   intros.
   split; auto.
   split with (\t, sub r (sublift rs)); auto.
   apply ax_lam_val; auto.
   intros.
      assert (RR (typ (t :: rhos) r) (sub r ((s :: rs) # shift rs))).
     apply
      (IHr ((s :: rs) # shift rs) (t :: rhos) (typ (t :: rhos) r)).
      simpl in |- *.
        split.
       trivial.
       trivial.
      simpl in H0.
        unfold TypJ in |- *.
        split.
       inversion H.
         auto.
       trivial.
     assert
      (Rd (typ (t :: rhos) r) (sub (\ t, r) rs; s)
         (sub r ((s :: rs) # shift rs))).
      apply ax_beta.
      apply Red_inv_preserves_RR with (s := sub r ((s :: rs) # shift rs)).
       apply TypJ_App with (rho := t).
        trivial.
        apply RR_implies_typcor.
          trivial.
       trivial.
       trivial.
(* R *)
  intros.
  simpl.
  destruct H.
  simpl in H0; subst.
  assert (TypJ rhos r1 Int).
  inversion H; auto.
  assert (TypJ rhos r2 (typ rhos r2)).
  inversion H; auto.
  assert (TypJ rhos r3 (Int-->typ rhos r2-->typ rhos r2)).
  split; auto.
  inversion H; subst; auto.
  inversion H; subst; auto.
  assert (h1:=IHr1 rs rhos Int X H0).
  assert (h2:=IHr2 rs rhos (typ rhos r2) X H1).
  assert (h3:=IHr3 rs rhos (Int-->typ rhos r2--> typ rhos r2) X H2).
  destruct h1.
  destruct y.
  apply R_aux with x; auto.
  destruct e; auto.
  destruct e; auto.
Focus 2.
  eapply IHr2; eauto.
  inversion H; auto.
Focus 2.
  eapply IHr3; eauto.
  inversion H; auto.

split; auto.
assert (length rhos<=length rs).
 assert (length rhos = length rs).
  apply RRs_length; auto.
  intuition.
inversion H; subst.
repeat constructor; auto;
destruct (RR_implies_typcor _ _ (IHr1 rs rhos Int X H0)); auto;
try elim RR_implies_typcor with (typ rhos r2) (sub r2 rs); auto;
try elim RR_implies_typcor with (Int-->typ rhos r2-->typ rhos r2) (sub r3 rs); auto.
Qed.


(** Main theorem **)

Lemma whnf : forall (r:term) (rho:type), TypJ nil r rho -> halts rho r.

Proof.
intros.
apply RR_implies_halt.
 trivial.
 assert (RR rho r).
 rewrite <- (sub_id r O).
 apply subst_preserves_RR with (@nil type); auto.
 unfold id; simpl; auto.
 auto.
Qed.

(* recovering type-checking in the extracted evaluator *)

Inductive Answer : Set := 
| No : Answer
| Yes : term -> Answer.

Inductive WTHalts : type -> term -> Answer -> Prop :=
| fail : forall rho r, ~TypJ nil r rho -> WTHalts rho r No
| success : forall rho r s, TypJ nil r rho -> Ev rho r s -> WTHalts rho r (Yes s).

Lemma typ_dec :
  forall r rhos rho, {TypJ rhos r rho}+{~TypJ rhos r rho}.
Proof.
induction r; intros.
case rho; [left | right]; auto.
intro abs; inversion abs.
simpl in H0; discriminate.
destruct (IHr rhos rho); intros.
generalize t; clear t; case rho; [left | right]; auto.
inversion t; repeat constructor; auto.
intro abs; inversion abs.
simpl in H0; discriminate.
right; intro abs.
inversion abs; subst.
inversion H; subst.
elim n; auto.

case (type_dec (nth n rhos dtype) rho); 
case (le_lt_dec (length rhos) n); simpl; intros; try discriminate.
right.
intro abs.
inversion abs.
inversion H; subst; auto with zarith.

left; constructor; auto;
right; intro abs; inversion abs;
inversion H; auto.

right; intro abs; inversion abs; inversion H; auto.
right; intro abs; inversion abs; inversion H; auto.

destruct (IHr2 rhos (typ rhos r2)); intros.
destruct (IHr1 rhos (typ rhos r2-->rho)).
left; apply TypJ_App with (typ rhos r2); auto.
right; intro abs; inversion abs; auto.
inversion H; subst.
apply n.
constructor; simpl; auto.
rewrite H6; auto.

right; intro abs; inversion abs; subst.
elim n.
inversion H; subst; auto.

case rho; simpl.
right; intro abs; inversion abs; subst.
simpl in H0; discriminate.
intros.
elim (type_dec t t0); intros.
destruct (IHr (t::rhos) t1).
left; repeat constructor; simpl; auto.
elim t2; auto.
inversion t2; congruence.
right; intro abs; inversion abs; elim n; auto.
constructor; auto.
inversion H; auto.
simpl in H0; auto.
injection H0; intros; subst; auto.
right; intro abs; inversion abs; subst.
simpl in H0; injection H0; intros; subst; auto.

elim IHr1 with rhos Int; elim IHr2 with rhos t; 
elim IHr3 with rhos (Int-->t-->t); intros;
elim (type_dec t rho); intros; subst; auto;
try (left; split; auto; destruct a1; destruct a; destruct a0; repeat constructor; auto; fail); 
right; intro abs; destruct abs; simpl in H0; inversion H; subst; auto.
Qed.

Lemma whnf_total : forall r rho, {a:Answer | WTHalts rho r a}.
Proof.
intros.
elim (typ_dec r nil rho); intros.
destruct (whnf a).
split with (Yes x); auto.
constructor; auto.
split with No; auto.
constructor 1; auto.
Qed.

End NormalizationProof.
