(*
#directory "../common";;
#load "../common/namedTerm.cmo";;
#use "04a.ml";;
*)

open NamedTerm

type 'a cache = 'a option ref

let cached_call (d : 'a cache) (t:unit -> 'a) : 'a =
  match !d with
  | Some y -> y
  | None   -> let y = t () in d := Some y; y

type value = Abs of identifier * (value -> value) | Inert of inert
  | Cache of n_term cache * value
and inert = V of identifier | IApp of inert * value
  | ICache of n_term cache * inert

let gensym : unit -> int =
  let c = ref 0 in
  fun () ->
    let res = !c in
    c := res + 1;
    res

let rec reify : value -> n_term =
  function
  | Abs (x, f) ->
    let xm = x ^ "_" ^ string_of_int (gensym ()) in
    NLam (xm, reify (f @@ Inert (V xm)))
  | Inert i -> render_inert i
  | Cache (c, v) -> cached_call c (fun () -> reify v)
and apply_value : value -> (value -> value) =
  function
  | Abs (x, f) -> f
  |           Inert i  -> apply_inert i
  | Cache (c, Inert i) -> apply_inert @@ ICache (c, i)
  | Cache (c,       v) -> apply_value v
and apply_inert (i : inert) (v' : value) : value =
  Inert (IApp (i, v'))
and render_inert : inert -> n_term =
  function
  | V x           -> NVar x
  | IApp (i, v)   -> let n = reify v in NApp (render_inert i, n)
  | ICache (c, i) -> cached_call c (fun () -> render_inert i)

module Dict = Map.Make(struct type t = identifier let compare = compare end)

type env = value Dict.t

let rec env_lookup (x : identifier) (e : env) : value =
  match Dict.find_opt x e with
  | Some v -> v
  | None   -> Inert (V (x ^ "_free"))

let mount_cache (v:value) : value =
  match v with
  | Cache(_,_) -> v
  | _          -> Cache(ref None, v)

let rec eval (t : n_term) (e : env) : value =
  match t with
  | NVar x        -> env_lookup x e
  | NLam (x, t')  -> Abs (x, apply_abs x t' e)
  | NApp (t1, t2) -> let v2 = eval t2 e
                     in apply_value (eval t1 e) v2
and apply_abs (x : identifier) (t : n_term) (e : env) (v : value) : value =
  eval t @@ Dict.add x (mount_cache v) e

let normal_form (t : n_term) : n_term = reify (eval t Dict.empty)

let _ = cbv_tests normal_form

