(*
#directory "../common";;
#load "../common/namedTerm.cmo";;
#use "01a_named.ml";;
*)

(* Here we modify the evaluator from 00_a_nbe.ml such that it no
   longer uses de Bruijn notation. Terms are now representad in a
   standard way, with named variables.

   We change the semantic domain accordingly: delayed neutral terms do
   not depend on de Bruijn level anymore.  *)
 

open NamedTerm

type sem = Abs of (sem -> sem) | Neutral of (unit -> n_term)

(*    Abstract variables are named *)
let abstract_variable (x : identifier) : sem =
  let vx = NVar x in
  Neutral (fun () -> vx)

(* A counter used for generation of fresh names *)
let gensym : unit -> int =
  let c = ref 0 in
  fun () ->
    let res = !c in
    c := res + 1;
    res

let rec reify : sem -> n_term =
  function
  | Abs f ->
    let xm = "x_" ^ string_of_int (gensym ()) in
    NLam (xm, reify (f @@ abstract_variable xm))
  | Neutral l ->
    l ()

let to_sem  (f : sem -> sem) : sem =  Abs f
  
let from_sem : sem -> (sem -> sem) =
  function
  | Abs f     -> f
  | Neutral l -> fun v' ->
    Neutral (fun () -> let n = reify v' in NApp (l (), n))

(* Environments now require names of variables *)                 
type env = (identifier * sem) list

let rec env_lookup (x : identifier) (e : env) : sem =
  match e with
  | []          -> abstract_variable (x ^ "_free")
  | (x', v)::e' -> if x = x' then v else env_lookup x e'

let rec eval (t : n_term) (e : env) : sem =
  match t with
  | NVar x        -> env_lookup x e
  | NLam (x, t')  -> to_sem  (fun v -> eval t' @@ (x, v)::e)
  | NApp (t1, t2) -> let v2 = eval t2 e
                     in from_sem (eval t1 e) v2

let nbe (t : n_term) : n_term = reify (eval t [])

let _ = cbv_tests nbe

