(*
#directory "../common";;
#load "../common/namedTerm.cmo";;
#use "01c_cache.ml";;
*)

(* Here we introduce caches to store information about shared
 *    computation. An α-cache is a place where a result of type α can be
 *    stored.  it can be used to prevent invoking the same delayed
 *    computation many times. *)

(* This way we obtain a reasonable higher-order evaluator that we
   later transform to an abstract machine. *) 

open NamedTerm

type 'a cache = 'a option ref

(* The result of delayed computation t is stored in cache d. When the
   cache contains a value, the computation is not repeated. *)              
let cached_call (c : 'a cache) (t : unit -> 'a) : 'a =
  match !c with
  | Some y -> y
  | None   -> let y = t () in c := Some y; y

(* there is one more kind of a value: when a normal form of a given
   value is already computed, it is stored in a cache *) 
type sem = Abs of (sem -> sem) | Neutral of (unit -> n_term)
  | Cache of n_term cache * sem

let abstract_variable (x : identifier) : sem =
  let vx = NVar x in
  Neutral (fun () -> vx)

let gensym : unit -> int =
  let c = ref 0 in
  fun () ->
    let res = !c in
    c := res + 1;
    res

(* The first two cases below are the same as in "01_env". In the third
   case we just memoize the result of reification.  *) 
let rec reify : sem -> n_term =
  function
  | Abs f ->
    let xm = "x_" ^ string_of_int (gensym ()) in
    NLam (xm, reify (f @@ abstract_variable xm))
  | Neutral l ->
    l ()
  | Cache (c, v) -> cached_call c (fun () -> reify v)

let to_sem  (f : sem -> sem) : sem =  Abs f
                  
(* Again, the first two cases below are not changed (the function
   "apply_neutral" is simply extracted). The main change is in the
   third case: l is the same as fun () -> l (); the latter one is
   optimized by caching the call l (). *)                  
let rec from_sem : sem -> (sem -> sem) =
  function
  | Abs f                -> f
  |           Neutral l  -> apply_neutral l
  | Cache (c, Neutral l) -> apply_neutral (fun () -> cached_call c l)
  | Cache (c,         v) -> from_sem v
and apply_neutral (l : unit -> n_term) (v : sem) : sem =
  Neutral (fun () -> let n = reify v in NApp (l (), n))

module Dict = Map.Make(struct type t = identifier let compare = compare end)

type env = sem Dict.t

let rec env_lookup (x : identifier) (e : env) : sem =
  match Dict.find_opt x e with
  | Some v -> v
  | None   -> abstract_variable (x ^ "_free")


(* The following is (the only) constructor of a cache. It avoids
   nested caches: we do not want to introduce new caches for 
   already cached values *)            
let mount_cache (v:sem) : sem =
  match v with
  | Cache(_,_) -> v
  | _          -> Cache(ref None, v)

(* The evaluation of variables and applications is standard. In the
   case of abstractions, there might be some shared computation stored
   in the environment; we capture this sharing by introducing caches*)                
let rec eval (t : n_term) (e : env) : sem =
  match t with
  | NVar x        -> env_lookup x e
  | NLam (x, t')  -> to_sem (fun v -> eval t' @@ Dict.add x (mount_cache v) e)
  | NApp (t1, t2) -> let v2 = eval t2 e
                     in from_sem (eval t1 e) v2

let nbe (t : n_term) : n_term = reify (eval t Dict.empty)

let _ = cbv_tests nbe

