(*
#directory "../common";;
#load "../common/namedTerm.cmo";;
#use "06a.ml";;
*)

(* Here we transform the evaluator to the continuation passing
   style. This is the first part of the transformation: we add the
   continuations and change the types of all functions, but these
   functions remain in the direct style. *)

open NamedTerm

type 'a cache = 'a option ref

module Dict = Map.Make(struct type t = identifier let compare = compare end)

type value = Abs of identifier * n_term * env | Inert of inert
  | Cache of n_term cache * value
and inert = V of identifier | IApp of inert * value
  | ICache of n_term cache * inert
and env = value Dict.t

let gensym : unit -> int =
  let c = ref 0 in
  fun () ->
    let res = !c in
    c := res + 1;
    res

let rec env_lookup (x : identifier) (e : env) : value =
  match Dict.find_opt x e with
  | Some v -> v
  | None   -> Inert (V (x ^ "_free"))

let mount_cache (v:value) : value =
  match v with
  | Cache(_,_) -> v
  | _          -> Cache(ref None, v)


let idk (x : 'a) : 'a = x

let rec eval (t : n_term) (e : env) (k : value -> 'a) : 'a =
  match t with
  | NVar x        -> k @@ env_lookup x e
  | NLam (x, t')  -> k @@ Abs (x, t', e)
  | NApp (t1, t2) -> k @@ let v2 = eval t2 e idk
                     in apply_value (eval t1 e idk) v2 idk
and apply_abs (x : identifier) (t : n_term) (e : env) (v : value) (k : value -> 'a) : 'a =
  k @@ eval t (Dict.add x (mount_cache v) e) idk
and reify (v : value) (k : n_term -> n_term) : n_term =
  match v with
  | Abs (x, t', e) ->
    let xm = x ^ "_" ^ string_of_int (gensym ()) in
    k @@ NLam (xm, reify (apply_abs x t' e (Inert (V xm)) idk) idk)
  | Inert i -> k @@ render_inert i idk
  | Cache (c, v) -> k @@ cached_reify c v idk
and cached_reify (d : n_term cache) (v : value) (k : n_term -> n_term) =
  match !d with
  | Some y -> k y
  | None   -> let y = reify v idk in d := Some y; k y
and apply_value (v : value) (v2 : value) (k : value -> 'a) =
  k @@ (match v with
  | Abs (x, t', e)     -> apply_abs x t' e v2 idk
  |           Inert i  -> apply_inert i v2
  | Cache (c, Inert i) -> apply_inert (ICache (c, i)) v2
  | Cache (c,       v) -> apply_value v v2 idk)
and apply_inert (i : inert) (v' : value) : value =
  Inert (IApp (i, v'))
and render_inert (i : inert) (k : n_term -> n_term) : n_term =
  match i with
  | V x           -> k @@ NVar x
  | IApp (i, v)   -> k @@ let n = reify v idk in NApp (render_inert i idk, n)
  | ICache (c, i) -> k @@ cached_reify c (Inert i) idk

let normal_form (t : n_term) : n_term = reify (eval t Dict.empty idk) idk

let _ = cbv_tests normal_form

