using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Collections;

namespace LogicImage
{

  // -----------------------------------------------------------
  // Game
  // -----------------------------------------------------------
  public class CGame 
  {
    // internal class to store the moves
    public class Move 
    { 
      int x, y;
      SquareClick click;

      public int X { get { return x; } }
      public int Y { get { return y; } }
      public SquareClick Click { get { return click; } }

      private Move() {}
      public Move( int x, int y, SquareClick click )
      {
        this.x = x;
        this.y = y;
        this.click = click;
      }
    }

    private Form fParent;

    private int gridSize;
    private int gridMargin;
    private int cellSizeX, cellSizeY;
    private string author      = string.Empty;
    private string description = string.Empty;

    public int GridSize    { get { return gridSize; } }
    public int GridMargin  { get { return gridMargin; } }
    public int CellSizeX   { get { return cellSizeX; } }
    public int CellSizeY   { get { return cellSizeY; } }
    public string Author      { get { return author; } }
    public string Description { get { return description; } }

    public  enum SquareState { None, Checked, Empty };
    public  enum SquareClick { Forward, Backward };

    public int[,]         RowData;
    public int[,]         ColData;
    public SquareState[,] TheBoard; 

    public ArrayList      Moves     = new ArrayList();
    public ArrayList      UndoMoves = new ArrayList();

    // -----------------------------------------------------------
    // constructor
    // -----------------------------------------------------------
    public CGame( int gridSize, int gridMargin, 
                  int cellSizeX, int cellSizeY,
                  string author, string description )
    {
      this.gridSize    = gridSize;
      this.gridMargin  = gridMargin;
      this.cellSizeX   = cellSizeX;
      this.cellSizeY   = cellSizeY;
      this.author      = author;
      this.description = description;

      TheBoard   = new SquareState[ GridSize, GridSize ];
      RowData    = new int[ GridSize, GridMargin ];
      ColData    = new int[ GridMargin, GridSize ];
    }
    
    // -----------------------------------------------------------
    // resize
    // -----------------------------------------------------------
    public void PrepareOwnerForm( Form fParent )
    {
      int sx = 2*CellSizeX + (GridSize+GridMargin) * CellSizeX;
      int sy = 2*CellSizeY + (GridSize+GridMargin) * CellSizeY;

      this.fParent = fParent;
 
      fParent.ClientSize = new Size( sx, sy );
      fParent.Text       = string.Format( "{0}, author:{1}", Description, Author );

      fParent.Invalidate();
    }

    // -----------------------------------------------------------
    // user activity
    // -----------------------------------------------------------
    public Rectangle GetCellRect( int row, int col )
    {
      return new Rectangle( CellSizeX+row*CellSizeX, 
                            CellSizeY+col*CellSizeY, 
                            CellSizeX, CellSizeY ); 
    }

    public Point GetCellFromPoint( int x, int y )
    {
      return new Point( (x-CellSizeX) / CellSizeX,
                        (y-CellSizeY) / CellSizeY );
    }
    public Point GetBoardPoint( Point p )
    {
      return new Point( p.X-GridMargin, p.Y-GridMargin );
    }
    public bool IsInsideBoard( Point p )
    {
      return p.X >= 0 && p.X < GridSize && p.Y >= 0 && p.Y < GridSize;
    }

    // -----------------------------------------------------------
    // undo, redo
    // -----------------------------------------------------------
    public void UndoMove()
    {
      if ( Moves.Count > 0 )
      {
        Move m = (Move)Moves[Moves.Count-1];
        Moves.RemoveAt( Moves.Count-1 );

        UndoMoves.Add( m );

        // click the square
        Rectangle r = GetCellRect( m.X, m.Y );
        for ( int i=0; i<(int)SquareState.Empty; i++ )
        {
          SquareClicked( r.X+1, r.Y+1, m.Click, false );
        }
      }
    }

    public void RedoMove()
    {
      if ( UndoMoves.Count > 0 )
      {
        Move m = (Move)UndoMoves[UndoMoves.Count-1];
        UndoMoves.RemoveAt( UndoMoves.Count-1 );

        // click the square
        Rectangle r = GetCellRect( m.X, m.Y );
        SquareClicked( r.X+1, r.Y+1, m.Click );
      }
    }

    // -----------------------------------------------------------
    // Square Clicked
    // -----------------------------------------------------------
    public void SquareClicked( int x, int y, SquareClick direction )
    {
      SquareClicked( x, y, direction, true );
    }
    public void SquareClicked( int x, int y, SquareClick direction, bool storeMove )
    {
      // process the move
      Point p = GetCellFromPoint( x, y );

      // board
      Point bp = GetBoardPoint( p );
      if ( IsInsideBoard( bp ) )
      {
        // store the click
        if ( storeMove ) Moves.Add( new Move( p.X, p.Y, direction ) );

        // perform the click
        int v = (int)TheBoard[bp.X, bp.Y];

        if ( direction == SquareClick.Forward )
        {
          v++; if ( v>(int)SquareState.Empty ) v=0;
        }
        else if ( direction == SquareClick.Backward )
        {
          v--; if ( v<(int)SquareState.None ) v=(int)SquareState.Empty;
        }
        TheBoard[bp.X, bp.Y] = (SquareState)v;
      }
      
      // margins
      if ( p.X >=0 && p.X < GridMargin && p.Y >= GridMargin && p.Y<GridMargin+GridSize )
        RowData[p.Y-GridMargin, p.X] = -RowData[p.Y-GridMargin, p.X];
      if ( p.Y >=0 && p.Y < GridMargin && p.X >= GridMargin && p.X<GridMargin+GridSize )
        ColData[p.Y, p.X-GridMargin] = -ColData[p.Y, p.X-GridMargin];

      if ( IsGameFinished() ) GameFinished();       

      //repaint
      if ( fParent != null ) fParent.Invalidate();
    }

    // -----------------------------------------------------------
    // game finished?
    // -----------------------------------------------------------
    bool IsGameFinished()
    {
      // rows
      for ( int r=0; r<GridSize; r++ )
      {
        // build
        ArrayList aBlocks = new ArrayList();

        int bSize = 0;
        for ( int c=0; c<GridSize; c++ )
        {
          if ( TheBoard[c,r] == SquareState.Checked ) 
            bSize++;
          else
          { 
            if ( bSize > 0 )
            {
              aBlocks.Add( bSize );
              bSize = 0;
            }
          }
        }
        if ( bSize > 0 ) aBlocks.Add( bSize );
        int aCount = aBlocks.Count;
        if ( aCount < GridMargin )
          for ( int i=0; i<GridMargin-aCount; i++ )
          {
            aBlocks.Insert( 0, 0 );
          }        
        // check
        for ( int c=0; c<GridMargin; c++ )
          if ( Math.Abs(RowData[r,c]) != (int)aBlocks[c] )
          {
            return false;
          }
      }

      // cols
      for ( int c=0; c<GridSize; c++ )
      {
        // build
        ArrayList aBlocks = new ArrayList();

        int bSize = 0;
        for ( int r=0; r<GridSize; r++ )
        {
          if ( TheBoard[c,r] == SquareState.Checked ) 
            bSize++;
          else
          { 
            if ( bSize > 0 )
            {
              aBlocks.Add( bSize );
              bSize = 0;
            }
          }
        }
        if ( bSize > 0 ) aBlocks.Add( bSize );
        int aCount = aBlocks.Count;
        if ( aCount < GridMargin )
          for ( int i=0; i<GridMargin-aCount; i++ )
          {
            aBlocks.Insert( 0, 0 );
          }        
        // check
        for ( int r=0; r<GridMargin; r++ )
          if ( Math.Abs(ColData[r,c]) != (int)aBlocks[r] )
          {
            return false;
          }
      }

      // ok
      return true;
    }

    void GameFinished()
    {
      MessageBox.Show( "Puzzle solved.", "Congratulations!", 
                       MessageBoxButtons.OK, 
                       MessageBoxIcon.Information );
    }

    // -----------------------------------------------------------
    // repaint
    // -----------------------------------------------------------
    public void PaintTheBoard( Graphics g )
    {
      // the info area rects
      g.FillRectangle( SystemBrushes.ControlLightLight, 
                       new Rectangle( CellSizeX*(1+GridMargin), CellSizeY, 
                                      CellSizeX*GridSize, CellSizeY*GridMargin ) );
      g.FillRectangle( SystemBrushes.ControlLightLight, 
                       new Rectangle( CellSizeX, CellSizeY*(1+GridMargin),  
                                      CellSizeX*GridMargin, CellSizeY*GridSize ) );

      // the info area 
      Pen pDash = new Pen( Color.Black );
      pDash.DashStyle = DashStyle.Dash;
      for ( int r=0; r<GridMargin+1; r++ )
      {
        g.DrawLine( pDash, 
                    new Point( CellSizeX*(1+GridMargin         ), CellSizeY*(r+1) ), 
                    new Point( CellSizeX*(1+GridMargin+GridSize), CellSizeY*(r+1) ) );  
        g.DrawLine( pDash, 
                    new Point( CellSizeX*(r+1), CellSizeY*(1+GridMargin         ) ), 
                    new Point( CellSizeX*(r+1), CellSizeY*(1+GridMargin+GridSize) ) );  
      }

      pDash.Dispose();
      
      // the grid
      for ( int r=0; r<GridSize+1; r++ )
        using ( Pen p = new Pen( Color.Black, (r%5==0 ? 2 : 1) ) )
        {
          g.DrawLine( p, 
                      new Point( CellSizeX                        , CellSizeY*(GridMargin+r+1) ), 
                      new Point( CellSizeX*(1+GridMargin+GridSize), CellSizeY*(GridMargin+r+1) ) );  
          g.DrawLine( p, 
                      new Point( CellSizeX*(GridMargin+r+1), CellSizeY                           ), 
                      new Point( CellSizeX*(GridMargin+r+1), CellSizeY*(1+GridMargin+GridSize) ) );  
        }

      // numbers
      StringFormat sf = new StringFormat();
      sf.Alignment    = sf.LineAlignment = StringAlignment.Center;

      using ( Font f = new Font( "Tahoma", (CellSizeY<12 ? 6 : CellSizeY-6) ) )
      {
        for ( int r=0; r<GridSize; r++ )
          for ( int c=0; c<GridMargin; c++ )
            if ( RowData[r, c]!=0 )
            {
              Rectangle rc = GetCellRect( c, r+GridMargin ); 
              g.DrawString( Math.Abs(RowData[r, c]).ToString(), f, Brushes.Black, 
                            rc, sf );                         

              if ( RowData[r,c]<0 ) DrawCross( g, Pens.Blue, rc );
            }
        for ( int r=0; r<GridMargin; r++ )
          for ( int c=0; c<GridSize; c++ )
            if ( ColData[r, c]!=0 )
            {
              Rectangle rc = GetCellRect( c+GridMargin, r );
              g.DrawString( Math.Abs(ColData[r, c]).ToString(), f, Brushes.Black, 
                            rc, sf );
              if ( ColData[r,c]<0 ) DrawCross( g, Pens.Blue, rc );
            }

      // paint the board state
      for ( int r=0; r<GridSize; r++ )
        for ( int c=0; c<GridSize; c++ )
        {
          if ( TheBoard[r,c] == SquareState.Checked )
          {
            Rectangle rc = GetCellRect( r+GridMargin, c+GridMargin );
            g.FillRectangle( Brushes.Black, rc );
            g.DrawRectangle( SystemPens.ControlLightLight, rc );
          }

          if ( TheBoard[r,c] == SquareState.Empty )
          {
            Rectangle cr = GetCellRect( r+GridMargin, c+GridMargin ); 
            DrawCross( g, Pens.Black, cr );
          }
        }

      }
    }

    void DrawCross( Graphics g, Pen p, Rectangle r )
    {
      g.DrawLine( p, new Point( r.X, r.Y ), new Point( r.X+r.Width, r.Y+r.Height ) );
      g.DrawLine( p, new Point( r.X+r.Width, r.Y ), new Point( r.X, r.Y+r.Height ) );
    }
  }
}