using System;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.DirectX;
using Microsoft.DirectX.Direct3D;

namespace DirectXTutorial
{
	public class DirectXForm : Form 
	{
		static string xFileName = null;
		Device	 device;
		Mesh	 mesh;
		int      meshParts = 0;
		ExtendedMaterial[] materials = null;
		Material material;
		float    rotationAngle = 0;
		PresentParameters pp;

		public DirectXForm() 
		{
			this.Size = new Size(300, 300);
			this.Text = "DirectX.NET";
		}

		bool InitializeGraphics() 
		{
			try
			{
				pp = new PresentParameters();
				pp.Windowed = true;
				pp.SwapEffect = SwapEffect.Discard;
				pp.EnableAutoDepthStencil = true;
				pp.AutoDepthStencilFormat = DepthFormat.D16;

				device = new Device(0, DeviceType.Hardware, this, CreateFlags.SoftwareVertexProcessing, pp);
				device.DeviceReset += new EventHandler(OnDeviceReset);

				InitializeD3DObjects();

				return true;
			}
			catch (DirectXException)
			{
				return false;
			}
		}

		void InitializeD3DObjects() 
		{
			CreateMesh();
			CreateMaterials();
			CreateLights();
			InitializeView();
		}

		void OnDeviceReset(object o, EventArgs e) 
		{
			InitializeD3DObjects();
		}

		protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
		{
			if ((int)(byte)e.KeyChar == (int)Keys.Escape)
				this.Close(); // zakocz 
		}

		void CreateMesh() 
		{
			//mesh = Mesh.Teapot( device );
			//meshParts = 1;

			mesh = Mesh.FromFile( xFileName, 0, device, out materials );
			meshParts = materials.Length;
		}

		void CreateMaterials() 
		{
			material = new Material();
			material.Ambient = Color.FromArgb( 0, 80, 80, 80);
			material.Diffuse = Color.FromArgb(0, 200, 200, 200);
			material.Specular = Color.FromArgb(0, 255, 255, 255);
			material.SpecularSharpness = 128.0f;
		}

		void CreateLights() 
		{
			Light light0 = device.Lights[0];
			Light light1 = device.Lights[1];

			light0.Type = LightType.Directional;
			light0.Direction = new Vector3(-1, 1, 5);
			light0.Diffuse = Color.White;
			light0.Enabled = true;
			light0.Commit();

			light1.Type = LightType.Spot;
			light1.Position  = new Vector3(-10, 10, -50);
			light1.Direction = new Vector3(10, -10, 50);
			light1.InnerConeAngle = 0.5f;
			light1.OuterConeAngle = 1.0f;
			light1.Diffuse		  = Color.LightBlue;
			light1.Specular		  = Color.White;
			light1.Range          = 1000.0f;
			light1.Falloff        = 1.0f;
			light1.Attenuation0   = 1.0f;
			light1.Enabled        = true;
			light1.Commit();

			device.RenderState.Lighting = true;
			device.RenderState.DitherEnable = false;
			device.RenderState.SpecularEnable = true;
			device.RenderState.Ambient = Color.FromArgb(0, 20, 20, 20);
		}

		void InitializeView() 
		{
			Vector3 eyePosition = new Vector3(0, 0, -40);
			Vector3 direction   = new Vector3(0, 0, 0);
			Vector3 upDirection = new Vector3(0, 1, 0);

			Matrix view = Matrix.LookAtLH(eyePosition, direction, upDirection );
			device.SetTransform(TransformType.View, view);

			float fieldOfView = (float)Math.PI/4;
			float aspectRatio = 1.0f;
			float nearPlane   = 1.0f;
			float farPlane    = 500.0f;

			Matrix projection = Matrix.PerspectiveFovLH(fieldOfView, aspectRatio, nearPlane, farPlane);
			device.SetTransform(TransformType.Projection, projection);
		}

		void AdvanceFrame() 
		{
			rotationAngle += 0.02f;
			rotationAngle %= Geometry.DegreeToRadian(360);

			Matrix rotateX = Matrix.RotationX(rotationAngle);
			Matrix rotateY = Matrix.RotationY(rotationAngle);
			Matrix world = Matrix.Multiply(rotateX, rotateY);
			device.SetTransform( TransformType.World, world );
		}

		void Render() 
		{
			device.Clear(ClearFlags.Target | ClearFlags.ZBuffer, Color.Black.ToArgb(), 1.0f, 0);
			device.BeginScene();
			device.Material = material;

			for ( int i=0; i<meshParts; i++ )
			{
			    device.Material = materials[i].Material3D;
				mesh.DrawSubset(i);
			}

			device.EndScene();
			device.Present();
		}

		public static void Main( string[] args ) 
		{
			using ( DirectXForm dxForm = new DirectXForm() )
			{
			    if ( args.Length > 0 )
				   xFileName = args[0];
				else
				   xFileName = "heli.x";

				if (!dxForm.InitializeGraphics()) 
				{
					MessageBox.Show( "Bd inicjowania Direct3D." );
					return;
				}
			
				dxForm.Show();

				DateTime start = DateTime.Now;
				int      frame = 0;
				while ( dxForm.Created )
				{
					frame++;
					dxForm.AdvanceFrame();
					dxForm.Render();

					dxForm.Text = String.Format( "FPS: {0:N}", frame/((TimeSpan)(DateTime.Now-start)).TotalSeconds );
					Application.DoEvents();
				}
			}
		}
	}
}
 